local UI = game.UI
local classlib = require("core.class")
local tablex = require("core.tablex")
local consts = require("ui.consts")
local colors = require("ui.colors")
local fsm = require("ui.fsm")
local lamsIDs = require("ui.lamsConsts")
local messageRequest = require("ui.messageRequest")
local pickupConsts = require("ui.pickupConsts")
local pickupUtil = require("ui.pickupUtil")
local queue = require("ui.queue")
local resourceConsts = require("ui.resourceConsts")
local resourceUtil = require("ui.resourceUtil")
local runeUtil = require("ui.runeUtil")
local sliderHoldFill = require("ui.sliderHoldFill")
local uiCalls = require("ui.uicalls")
local util = require("ui.util")
local DesignerMessage = classlib.Class("DesignerMessage", fsm.UIState)
local DMTextOff = DesignerMessage:StateClass("DMTextOff", fsm.UIState)
local DMTextOn = DesignerMessage:StateClass("DMTextOn", fsm.UIState)
local DMAdvancePressButton = DesignerMessage:StateClass("DMAdvancePressButton", fsm.UIState)
local DMAdvanceHoldButton = DesignerMessage:StateClass("DMAdvanceHoldButton", fsm.UIState)
local DMTextSuspendHelper = DesignerMessage:StateClass("DMTextSuspendHelper", fsm.UIState)
local useGrey = false
local useGreyAnimation = false
local designerMessage = DesignerMessage.New("designerMessage", {
  DMTextOff,
  DMTextOn,
  {DMAdvancePressButton, DMAdvanceHoldButton},
  DMTextSuspendHelper
})
function DesignerMessage:SetStreamingAlgorithm(value)
  if value == nil then
    value = false
  end
  UI.SetForcePauseMenuStreamingMode(value)
end
function DesignerMessage:Idle()
  local kratos = game.Player.FindPlayer()
  if kratos ~= nil and kratos.Pad.UIDisableHack ~= nil then
    kratos.Pad:UIDisableHack(true, consts.UIBUTTONDISABLE_DESIGNERMESSAGE)
  end
  UI.Idle(true, true, true)
  self.isIdling = true
end
function DesignerMessage:UnIdle()
  local kratos = game.Player.FindPlayer()
  if kratos ~= nil and kratos.Pad.UIDisableHack ~= nil then
    kratos.Pad:UIDisableHack(false, consts.UIBUTTONDISABLE_DESIGNERMESSAGE)
  end
  UI.Idle(false, false, true)
  self.isIdling = false
end
function DesignerMessage:Setup()
  self.messageQueue = queue.Queue.New()
  self.currentMessage = nil
  self.queueMax = 20
  self.promptFadeInTime = 0.2
  self.promptFadeOutTime = 0.2
  self.promptAnimSpeed = 5
  self.gameIsPaused = false
  self.wantedPadEvents = false
  self.pauseMenuOpen = false
  self.isIdling = false
  self.goDesignerMessage = util.GetUiObjByName("DesignerMessage")
  self.goHUDTutorialMessage = util.GetUiObjByName("HUDTutorialMessage")
  self.goItemCardMessage = util.GetUiObjByName("ItemCardMessage")
  self.goSplashScreenMessage = util.GetUiObjByName("SplashScreenMessage_LandscapeSmall")
  self.goSidebarMessageSmall = util.GetUiObjByName("SidebarTutorialMessage_Small")
  self.goSidebarMessageMed = util.GetUiObjByName("SidebarTutorialMessage_Medium")
  self.goSidebarMessageLarge = util.GetUiObjByName("SidebarTutorialMessage_Large")
  self.goBottomMessage = util.GetUiObjByName("BottomTutorialMessage")
  self.goSplashScreenMessage_LandscapeLarge = util.GetUiObjByName("SplashScreenMessage_LandscapeLarge")
  self.goSplashScreenMessage_PortraitSmall = util.GetUiObjByName("SplashScreenMessage_PortraitSmall")
  self.goSplashScreenMessage_PortraitLarge = util.GetUiObjByName("SplashScreenMessage_PortraitLarge")
  self.goSplashScreenMessage_NoImage = util.GetUiObjByName("SplashScreenMessage_NoImage")
  self.disabledButtons = {}
  local setDefaultText = function(messageRequest)
    self:SetStandardMessage(messageRequest)
  end
  local setTutorialText = function(messageRequest)
    self:SetTutorialMessage(messageRequest)
  end
  local setItemCardText = function(messageRequest)
    self:SetItemCardMessage(messageRequest)
  end
  local GetFadeGOs = function(currentMessage)
    return {
      currentMessage:get_goTextObjects()
    }
  end
  local GetItemCardFadeGOs = function(currentMessage)
    local goItemCardMessageChild = currentMessage:get_goMessageRequestChild()
    return {
      goItemCardMessageChild:FindSingleGOByName("TextObjects")
    }
  end
  local defaultPrepared = function(currentMessage)
    return true
  end
  local imagePrepared = function(currentMessage)
    if currentMessage:get_itemCardArgs() ~= nil then
      local go = currentMessage:get_goMessageRequestChild()
      local skipHighMips = game.CHECK_FEATURE("SAFE_SKIP_HIGH_MIPS")
      if not currentMessage._imagePreparedBeganLoad_ then
        self:SetStreamingAlgorithm(true)
        currentMessage._imagePreparedBeganLoad_ = true
        go:SetMaterialSwap(currentMessage:get_itemCardArgs())
        if go.RequestLoadHighMips ~= nil then
          go:RequestLoadHighMips(skipHighMips)
        end
        return false
      elseif go.RequestLoadHighMips ~= nil then
        go:RequestLoadHighMips(skipHighMips)
        return go:CheckMaterialFullyStreamed()
      end
    end
    return true
  end
  local onExitTransitionBegin = function(currentMessage)
    local fadeGOs = self.messageTypeDefaults[currentMessage:get_messageType()].GetFadeGOs(currentMessage)
    currentMessage:AnimateOut(fadeGOs, true)
  end
  local imageExitTransitionBegin = function(currentMessage)
    onExitTransitionBegin(currentMessage)
    self:SetStreamingAlgorithm(false)
  end
  local defaultMessageComparison = function(currentMessage, nextMessage)
    return currentMessage:get_messageType() == nextMessage:get_messageType()
  end
  local itemCardMessageComparison = function(currentMessage, nextMessage)
    return false
  end
  self.messageTypeDefaults = {
    [uiCalls.msgParam.MESSAGE_STANDARD] = {
      gameObject = self.goDesignerMessage,
      defaultExitDelay = 0,
      SetText = setDefaultText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = defaultPrepared,
      OnExitTransitionBegin = onExitTransitionBegin,
      CompareMessages = defaultMessageComparison
    },
    [uiCalls.msgParam.MESSAGE_TUTORIAL] = {
      gameObject = self.goHUDTutorialMessage,
      defaultExitDelay = 1,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = defaultPrepared,
      OnExitTransitionBegin = onExitTransitionBegin,
      CompareMessages = defaultMessageComparison
    },
    [uiCalls.msgParam.MESSAGE_BOTTOM] = {
      gameObject = self.goBottomMessage,
      defaultExitDelay = 1,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = defaultPrepared,
      OnExitTransitionBegin = onExitTransitionBegin,
      CompareMessages = defaultMessageComparison,
      EnterSound = "SND_UX_HUD_Item_Notification_Tip_Message_Bottom"
    },
    [uiCalls.msgParam.MESSAGE_SIDEBAR] = {
      gameObject = self.goSidebarMessageSmall,
      defaultExitDelay = 1,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = defaultPrepared,
      OnExitTransitionBegin = onExitTransitionBegin,
      OverrideAnimSpeed = 1.5,
      CompareMessages = defaultMessageComparison,
      EnterSound = "SND_UX_HUD_Item_Notification_Tip_Message_Side_Bar"
    },
    [uiCalls.msgParam.SIDEBAR_MEDIUM] = {
      gameObject = self.goSidebarMessageMed,
      defaultExitDelay = 1,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = defaultPrepared,
      OnExitTransitionBegin = onExitTransitionBegin,
      OverrideAnimSpeed = 1.5,
      CompareMessages = defaultMessageComparison,
      EnterSound = "SND_UX_HUD_Item_Notification_Tip_Message_Side_Bar"
    },
    [uiCalls.msgParam.SIDEBAR_LARGE] = {
      gameObject = self.goSidebarMessageLarge,
      defaultExitDelay = 1,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = defaultPrepared,
      OnExitTransitionBegin = onExitTransitionBegin,
      OverrideAnimSpeed = 1.5,
      CompareMessages = defaultMessageComparison,
      EnterSound = "SND_UX_HUD_Item_Notification_Tip_Message_Side_Bar"
    },
    [uiCalls.msgParam.MESSAGE_SPLASHSCREEN] = {
      gameObject = self.goSplashScreenMessage,
      defaultExitDelay = 0,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = imagePrepared,
      OnExitTransitionBegin = imageExitTransitionBegin,
      OverrideAnimSpeed = -1,
      OverrideFadeInTime = 0,
      OverrideFadeOutTime = 0,
      CompareMessages = defaultMessageComparison,
      OverrideDisplayTime = 1.1,
      EnterSound = "SND_UX_Tutorial_Prompt_Enter",
      ExitSound = "SND_UX_Tutorial_Prompt_Exit",
      ContinueSound = "SND_UX_HUD_Tutorial_Prompt_Tick",
      UseUIAccessibilityScaling = false
    },
    [uiCalls.msgParam.SPLASHSCREEN_LANDSCAPE_LARGE] = {
      gameObject = self.goSplashScreenMessage_LandscapeLarge,
      defaultExitDelay = 0,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = imagePrepared,
      OnExitTransitionBegin = imageExitTransitionBegin,
      OverrideAnimSpeed = -1,
      OverrideFadeInTime = 0,
      OverrideFadeOutTime = 0,
      CompareMessages = defaultMessageComparison,
      OverrideDisplayTime = 1.1,
      EnterSound = "SND_UX_Tutorial_Prompt_Enter",
      ExitSound = "SND_UX_Tutorial_Prompt_Exit",
      ContinueSound = "SND_UX_HUD_Tutorial_Prompt_Tick",
      UseUIAccessibilityScaling = false
    },
    [uiCalls.msgParam.SPLASHSCREEN_PORTRAIT_SMALL] = {
      gameObject = self.goSplashScreenMessage_PortraitSmall,
      defaultExitDelay = 0,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = imagePrepared,
      OnExitTransitionBegin = imageExitTransitionBegin,
      OverrideAnimSpeed = -1,
      OverrideFadeInTime = 0,
      OverrideFadeOutTime = 0,
      CompareMessages = defaultMessageComparison,
      OverrideDisplayTime = 1.1,
      EnterSound = "SND_UX_Tutorial_Prompt_Enter",
      ExitSound = "SND_UX_Tutorial_Prompt_Exit",
      ContinueSound = "SND_UX_HUD_Tutorial_Prompt_Tick",
      UseUIAccessibilityScaling = false
    },
    [uiCalls.msgParam.SPLASHSCREEN_PORTRAIT_LARGE] = {
      gameObject = self.goSplashScreenMessage_PortraitLarge,
      defaultExitDelay = 0,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = imagePrepared,
      OnExitTransitionBegin = imageExitTransitionBegin,
      OverrideAnimSpeed = -1,
      OverrideFadeInTime = 0,
      OverrideFadeOutTime = 0,
      CompareMessages = defaultMessageComparison,
      OverrideDisplayTime = 1.1,
      EnterSound = "SND_UX_Tutorial_Prompt_Enter",
      ExitSound = "SND_UX_Tutorial_Prompt_Exit",
      ContinueSound = "SND_UX_HUD_Tutorial_Prompt_Tick",
      UseUIAccessibilityScaling = false
    },
    [uiCalls.msgParam.SPLASHSCREEN_NO_IMAGE] = {
      gameObject = self.goSplashScreenMessage_NoImage,
      defaultExitDelay = 0,
      SetText = setTutorialText,
      GetFadeGOs = GetFadeGOs,
      PreparedToDisplay = defaultPrepared,
      OnExitTransitionBegin = imageExitTransitionBegin,
      OverrideAnimSpeed = -1,
      OverrideFadeInTime = 0,
      OverrideFadeOutTime = 0,
      CompareMessages = defaultMessageComparison,
      OverrideDisplayTime = 1.1,
      EnterSound = "SND_UX_Tutorial_Prompt_Enter",
      ExitSound = "SND_UX_Tutorial_Prompt_Exit",
      ContinueSound = "SND_UX_HUD_Tutorial_Prompt_Tick",
      UseUIAccessibilityScaling = false
    },
    [uiCalls.msgParam.MESSAGE_ITEM_CARD] = {
      gameObject = self.goItemCardMessage,
      defaultExitDelay = 1 / 4,
      SetText = setItemCardText,
      GetFadeGOs = GetItemCardFadeGOs,
      PreparedToDisplay = defaultPrepared,
      OnExitTransitionBegin = onExitTransitionBegin,
      OverrideAnimSpeed = 1 * 4,
      OverrideExitAnimSpeed = 1 * 4,
      OverrideFadeInTime = 0.35 / 4,
      OverrideFadeOutTime = 0.35 / 4,
      OverrideFadeInDelay = 1.65 / 4,
      AnimateFadeGOs = false,
      CompareMessages = itemCardMessageComparison,
      OverrideDisplayTime = 1 / 4,
      EnterSound = "SND_UX_HUD_Card_Prompt_Appear",
      ExitSound = "SND_UX_HUD_Card_Prompt_Dissolve",
      UseUIAccessibilityScaling = true
    }
  }
  self.advanceTypeInfo = {
    [uiCalls.msgParam.ADVANCE_PRESS] = {
      AdvanceState = "DMAdvancePressButton",
      PausesGame = false,
      CanBeDelayed = true,
      MasksButtons = true,
      PausesForCombat = true
    },
    [uiCalls.msgParam.ADVANCE_HOLD] = {
      AdvanceState = "DMAdvanceHoldButton",
      PausesGame = true,
      CanBeDelayed = false,
      MasksButtons = false
    },
    [uiCalls.msgParam.ADVANCE_TIMER] = {
      AdvanceState = "DMTextOff",
      PausesGame = false,
      CanBeDelayed = true,
      MasksButtons = false,
      PausesForCombat = false
    },
    [uiCalls.msgParam.ADVANCE_PRESS_NOPAUSE] = {
      AdvanceState = "DMAdvancePressButton",
      PausesGame = false,
      CanBeDelayed = true,
      MasksButtons = true,
      PausesForCombat = false
    },
    [uiCalls.msgParam.ADVANCE_PRESS_FORCEPAUSE] = {
      AdvanceState = "DMAdvancePressButton",
      PausesGame = true,
      CanBeDelayed = true,
      MasksButtons = false
    }
  }
  self.itemCardHelper = self:GenerateItemCardHelperTable()
  self.fadeDecal = util.GetUiObjByName("FadeGameOutDecal"):FindSingleGOByName("tutorialWorld")
  self.fadeDecal:Hide()
  self.defaultFooterInfo = {
    {
      Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Close),
      EventHandlers = {
        {
          Events = {
            "EVT_Advance_Release"
          },
          Handler = nil
        }
      }
    }
  }
  self.continueFooterInfo = {
    {
      Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Continue),
      EventHandlers = {
        {
          Events = {
            "EVT_Advance_Release"
          },
          Handler = nil
        }
      }
    }
  }
end
function DesignerMessage:Enter()
  self:Goto("DMTextOff")
  self:StartTimer("DesignerMessageInitializationTimer", 0.1, function()
    self.inputAvailable = true
    if not self.messageQueue:IsEmpty() then
      self:Goto("DMTextOn")
    end
  end)
end
function DesignerMessage:CurrentlyShowingMessage()
  return self.currentMessage ~= nil
end
function DesignerMessage:PauseGame(truePause)
  self:SendEventUI("EVT_DM_PAUSE", truePause)
  if useGrey then
    self.fadeDecal:Show()
    if not self.gameIsPaused then
      if useGreyAnimation then
        self.fadeDecal:JumpAnimationToFrame(0, {Animation = "default"})
        self.fadeDecal:PlayAnimationToEnd({Animation = "default", Rate = 2})
      else
        self.fadeDecal:JumpAnimationToPercent(1, {Animation = "default"})
      end
    end
  end
  self.gameIsPaused = true
end
function DesignerMessage:UnpauseGame()
  self:SendEventUI("EVT_DM_UNPAUSE")
  if useGrey and useGreyAnimation then
    self.fadeDecal:PlayAnimationToEnd({Animation = "default", Rate = -2})
    self:StartTimer("HideDecalTimer", 0.5, function()
      self.fadeDecal:Hide()
    end)
  else
    self.fadeDecal:Hide()
  end
  self.gameIsPaused = false
end
function DesignerMessage:EVT_TEXT_DESIGNER_MESSAGE(argTable)
  assert(argTable ~= nil and type(argTable) == "table", "Recieved an invalid table as a designer message.")
  assert(argTable.Text ~= nil and argTable.Text ~= "", "Recieved a nil or empty string as a designer message.")
  if self.messageQueue:GetCount() >= self.queueMax then
    self:DebugPrint("Designer Message queue maxed, ignoring newest message:" .. argTable.Text)
    return
  end
  self:SetupMessage(argTable)
  if not self.messageQueue:IsEmpty() and not self:CurrentlyShowingMessage() and self.inputAvailable then
    self:Goto("DMTextOn")
  end
end
function GetEntranceDelay(advanceTypeTable, curEntranceDelay)
  if curEntranceDelay ~= nil then
    return curEntranceDelay
  end
  if advanceTypeTable ~= nil and advanceTypeTable.PausesGame and useGrey and useGreyAnimation then
    return 0.5
  end
  return 0
end
function DesignerMessage:SetupMessage(argTable)
  local bodyMessage = argTable.Text
  local headerMessage = argTable.Title or ""
  local messageType = argTable.MessageType or uiCalls.msgParam.MESSAGE_STANDARD
  local defaultMessageTable = self.messageTypeDefaults[messageType]
  assert(defaultMessageTable ~= nil, "Designer Message recieved unknown message type!")
  local advanceType = argTable.AdvanceType or uiCalls.msgParam.ADVANCE_TIMER
  local thisAdvanceTypeInfo = self.advanceTypeInfo[advanceType]
  local entranceDelay = GetEntranceDelay(thisAdvanceTypeInfo, argTable.EntranceDelay)
  local displayTime = argTable.DisplayTime or defaultMessageTable.OverrideDisplayTime or 3.5
  local exitDelay = argTable.ExitDelay or defaultMessageTable.defaultExitDelay
  local animSpeed = argTable.AnimSpeed or defaultMessageTable.OverrideAnimSpeed or self.promptAnimSpeed
  local exitAnimSpeed = argTable.ExitAnimSpeed or defaultMessageTable.OverrideExitAnimSpeed or animSpeed
  local fadeInTime = argTable.FadeInTime or defaultMessageTable.OverrideFadeInTime or self.promptFadeInTime
  local fadeOutTime = argTable.FadeOutTime or defaultMessageTable.OverrideFadeOutTime or self.promptFadeOutTime
  local fadeInDelay = argTable.FadeInDelay or defaultMessageTable.OverrideFadeInDelay or 0
  local footerInfo = argTable.FooterButtonInfo or self.defaultFooterInfo
  local continueFooterInfo = argTable.ContinueFooterButtonInfo or argTable.FooterButtonInfo or self.continueFooterInfo
  local pausesGame = false
  local completionHook = argTable.CompletionHook
  local optionsOverride = argTable.OptionsOverride
  local useUIAccessibilityScaling = argTable.UseUIAccessibilityScaling or defaultMessageTable.UseUIAccessibilityScaling
  local maxUIAccessibilityScaling = argTable.MaxUIAccessibilityScaling or defaultMessageTable.MaxUIAccessibilityScaling
  if thisAdvanceTypeInfo ~= nil then
    pausesGame = thisAdvanceTypeInfo.PausesGame or false
  end
  local newMessage = messageRequest.MessageRequest.New(self, {
    MessageObject = defaultMessageTable.gameObject,
    BodyMessage = bodyMessage,
    HeaderMessage = headerMessage,
    Time = displayTime,
    MessageDelay = entranceDelay,
    ExitDelay = exitDelay,
    AdvanceType = advanceType,
    MessageType = messageType,
    AnimSpeed = animSpeed,
    FadeInTime = fadeInTime,
    FadeOutTime = fadeOutTime,
    FadeInDelay = fadeInDelay,
    ItemCardArgs = argTable.TypeSpecificData,
    TerminateCallback = argTable.TerminateCallback,
    ClearedOutCallback = argTable.ClearedOutCallback,
    CompletionHook = completionHook,
    FooterButtonInfo = footerInfo,
    ContinueFooterButtonInfo = continueFooterInfo,
    FormattedArgs = argTable.formattedArgs,
    TextStyle = argTable.TextStyle,
    UnpauseWhenComplete = not pausesGame,
    ExitAnimSpeed = exitAnimSpeed,
    AnimateFadeGOs = defaultMessageTable.AnimateFadeGOs,
    EnterSound = defaultMessageTable.EnterSound,
    ExitSound = defaultMessageTable.ExitSound,
    ContinueSound = defaultMessageTable.ContinueSound,
    OptionsOverride = optionsOverride,
    AdvanceToNextMessage = function()
      self:AdvanceToNextMessage()
    end,
    OnExitTransitionBegin = defaultMessageTable.OnExitTransitionBegin,
    UseUIAccessibilityScaling = useUIAccessibilityScaling,
    MaxUIAccessibilityScaling = maxUIAccessibilityScaling
  })
  if self:CurrentlyShowingMessage() and self.currentMessage:get_advanceType() == uiCalls.msgParam.INFINITE_TIMER then
    self:ClearAllMessages()
  end
  if argTable.MessageType == uiCalls.msgParam.MESSAGE_ITEM_CARD then
    self:ItemCardPrioritization(newMessage)
  elseif (game.UI.GetUserSuppressDesignerMessages and game.UI.GetUserSuppressDesignerMessages() or game.GetNewGamePlus()) and not argTable.Critical then
    newMessage:CallCompletionHook()
    local terminateCallback = newMessage:get_terminateCallback()
    if terminateCallback ~= nil then
      terminateCallback()
    end
  else
    self.messageQueue:Enqueue(newMessage)
  end
end
function DesignerMessage:ItemCardPrioritization(newMessage)
  if not self:HaveTimer("DMTimer") and self.currentMessage ~= nil and self.currentMessage:get_messageType() ~= uiCalls.msgParam.MESSAGE_ITEM_CARD and self.currentMessage:get_unpauseWhenComplete() and not self.isIdling then
    if self:HaveTimer("MessageDelayTimer") then
      self:StopTimer("MessageDelayTimer")
    end
    if self:HaveTimer("AdvanceHelperTimer") then
      self.currentMessage._time = self:GetTimerRemainingTime("AdvanceHelperTimer")
      self:StopTimer("AdvanceHelperTimer")
      self.currentMessage._designerMessages_EqualsPreviousMessage = true
      self.currentMessage._messageDelay = 0
    end
    self.messageQueue:EnqueueFront(self.currentMessage)
    self.currentMessage = nil
    self:Goto("DMTextSuspendHelper")
  end
  self.messageQueue:EnqueueFront(newMessage)
end
function DesignerMessage:OnMessageComplete()
  if game.CHECK_FEATURE("DISABLE_ITEM_CARDS") and engine.VFSGetBool("/UI/Disable Item Acquire") == 1 and not self.messageQueue:IsEmpty() and self.messageQueue:Peek():get_messageType() == uiCalls.msgParam.MESSAGE_ITEM_CARD then
    while not self.messageQueue:IsEmpty() and self.messageQueue:Peek():get_messageType() == uiCalls.msgParam.MESSAGE_ITEM_CARD do
      self.messageQueue:Dequeue()
    end
    if self.messageQueue:IsEmpty() or self.messageQueue:Peek():get_unpauseWhenComplete() then
      self:UnpauseGame()
    end
  end
  self:SetStreamingAlgorithm(false)
  self:UnIdle()
  if not self.messageQueue:IsEmpty() and not self.pauseMenuOpen and self.inputAvailable then
    self:Goto("DMTextOn")
  end
end
function DesignerMessage:EVT_CLEAR_ALL_DESIGNER_MESSAGES()
  self:ClearAllMessages()
end
function DesignerMessage:ClearAllMessages()
  self:Goto("DMTextSuspendHelper")
  if self.currentMessage ~= nil then
    self.currentMessage:CallCompletionHook()
  end
  if self:HaveTimer("MessageDelayTimer") then
    self:StopTimer("MessageDelayTimer")
  end
  if self:HaveTimer("AdvanceHelperTimer") then
    self:StopTimer("AdvanceHelperTimer")
  end
  if self:HaveTimer("DMTimer") then
    if self.gameIsPaused then
      self:UnpauseGame()
    end
    self.currentMessage = nil
    self:StopTimer("DMTimer")
  end
  while not self.messageQueue:IsEmpty() do
    local message = self.messageQueue:Dequeue()
    local clearedOutCallback = message:get_clearedOutCallback()
    if clearedOutCallback ~= nil then
      clearedOutCallback()
    end
    message:CallCompletionHook()
  end
  self:SetStreamingAlgorithm(false)
  self:AdvanceToNextMessage()
end
function DesignerMessage:AdvanceToNextMessage()
  self:Goto("DMTextOff")
end
function DesignerMessage:MessageDelayInternal()
  local tickTime = 0.02
  if self.currentMessage ~= nil then
    local messageType = self.currentMessage:get_messageType()
    local thisMessageTypeInfo = self.messageTypeDefaults[messageType]
    if not self.pauseMenuOpen and thisMessageTypeInfo.PreparedToDisplay(self.currentMessage) then
      self:DisplayMessage()
    else
      self:StartTimer("MessagePreparedTimer", tickTime, function()
        self:MessageDelayInternal()
      end)
    end
  end
end
function DesignerMessage:HandleMessageDelay()
  local messageDelay = self.currentMessage:get_messageDelay()
  local advanceType = self.currentMessage:get_advanceType()
  local thisAdvanceTypeInfo = self.advanceTypeInfo[advanceType]
  if thisAdvanceTypeInfo ~= nil then
    local pauseForCombat = thisAdvanceTypeInfo.PausesForCombat == true and game.Combat.GetCombatStatus()
    local truePause = false
    local needsPauseEvent = thisAdvanceTypeInfo.PausesGame or pauseForCombat or thisAdvanceTypeInfo.MasksButtons
    if thisAdvanceTypeInfo.PausesGame or pauseForCombat then
      truePause = true
    end
    if thisAdvanceTypeInfo.MasksButtons then
      self:Idle()
    end
    if needsPauseEvent then
      self:PauseGame(truePause)
    end
  end
  if messageDelay <= 0 then
    self:MessageDelayInternal()
  else
    self:StartTimer("MessageDelayTimer", messageDelay, function()
      self:MessageDelayInternal()
    end)
  end
end
function DesignerMessage:HandleMessageAdvance()
  local messageTime = self.currentMessage:get_time()
  local fullDelayTime = messageTime
  local advanceType = self.currentMessage:get_advanceType()
  local thisAdvanceTypeInfo = self.advanceTypeInfo[advanceType]
  if thisAdvanceTypeInfo ~= nil then
    if fullDelayTime == 0 or not thisAdvanceTypeInfo.CanBeDelayed then
      self:Goto(thisAdvanceTypeInfo.AdvanceState)
    else
      self:StartTimer("AdvanceHelperTimer", fullDelayTime, function()
        self:Goto(thisAdvanceTypeInfo.AdvanceState)
      end)
    end
  end
end
function DesignerMessage:HandleMessageExit()
  if self.currentMessage ~= nil then
    self.currentMessage:CallCompletionHook()
    if self.currentMessage:get_optionsOverride() ~= nil then
      self:SendEventToUIFsm("globalMenu", "EVT_SET_OPTIONS_OVERRIDE", nil)
    end
    local nextMessage = self.messageQueue:Peek()
    if nextMessage ~= nil and self.gameIsPaused and self.messageTypeDefaults[self.currentMessage:get_messageType()].CompareMessages(self.currentMessage, nextMessage) then
      nextMessage._designerMessages_EqualsPreviousMessage = true
      self.currentMessage:PlayContinueSound()
      self:OnMessageComplete()
      return
    end
    self.currentMessage:PlayExitSound()
    self.currentMessage:OnExitTransitionBegin()
    local exitDelay = self.currentMessage:get_exitDelay()
    local terminateCallback = self.currentMessage:get_terminateCallback()
    if terminateCallback ~= nil then
      terminateCallback()
    end
    if exitDelay == 0 then
      self:ClearScreen()
    else
      self:StartTimer("DMTimer", exitDelay, function()
        self:ClearScreen()
      end)
    end
  else
    self:ClearScreen()
  end
end
function DesignerMessage:ClearScreen()
  local nextMessage = self.messageQueue:Peek()
  for _, defaultTable in pairs(self.messageTypeDefaults) do
    if not (nextMessage ~= nil and nextMessage._designerMessages_EqualsPreviousMessage) or defaultTable.gameObject ~= nextMessage:get_goMessageRequest() then
      defaultTable.gameObject:Hide()
    end
  end
  if self.currentMessage ~= nil and self.gameIsPaused then
    local shouldUnpauseGame = true
    if nextMessage ~= nil then
      local nextAdvanceTypeInfo = self.advanceTypeInfo[nextMessage:get_advanceType()]
      if nextAdvanceTypeInfo ~= nil then
        local pauseForCombat = nextAdvanceTypeInfo.PausesForCombat == true and game.Combat.GetCombatStatus()
        local usePauseEvent = nextAdvanceTypeInfo.PausesGame or pauseForCombat or nextAdvanceTypeInfo.MasksButtons
        shouldUnpauseGame = not usePauseEvent
      end
    end
    if shouldUnpauseGame then
      self:UnpauseGame()
    end
  end
  self:UnIdle()
  self.currentMessage = nil
  self:OnMessageComplete()
end
function DesignerMessage:DisplayMessage()
  local messageType = self.currentMessage:get_messageType()
  local messageDefaults = self.messageTypeDefaults[messageType]
  messageDefaults.SetText(self.currentMessage)
  if self.currentMessage._designerMessages_EqualsPreviousMessage ~= true then
    self.currentMessage:PlayEnterSound()
    self.currentMessage:Show()
    self.currentMessage:AnimateIn(messageDefaults.GetFadeGOs(self.currentMessage), true)
  end
  if self.currentMessage:get_optionsOverride() ~= nil then
    self:SendEventToUIFsm("globalMenu", "EVT_SET_OPTIONS_OVERRIDE", self.currentMessage:get_optionsOverride())
  end
  self:HandleMessageAdvance()
end
function DesignerMessage:SetStandardMessage(messageRequest)
  assert(messageRequest ~= nil)
  local thBody = messageRequest:GetTextHandle("Body")
  messageRequest:SetText(thBody, messageRequest:get_bodyMessage(), true)
end
function DesignerMessage:SetTutorialMessage(messageRequest)
  assert(messageRequest ~= nil)
  local thHeader = messageRequest:GetTextHandle("Header")
  messageRequest:SetText(thHeader, messageRequest:get_headerMessage(), false)
  local thBody = messageRequest:GetTextHandle("Body")
  messageRequest:SetText(thBody, messageRequest:get_bodyMessage(), true)
end
function DesignerMessage:GenerateItemCardHelperTable()
  local RarityIndexToAnimTime = function(index)
    return (index - 1) * 0.1
  end
  local GetRuneDisplayInfo = function(runeID)
    local runeInfo = runeUtil.GetRuneInfo(runeID)
    local rarity, rarityIndex = runeUtil.GetRarityFromID(runeID)
    local typeName = util.GetLAMSMsg(lamsIDs.Enchantment)
    local perkId = runeInfo.Perks[1]
    local perkDescription
    if perkId ~= nil then
      perkDescription = tonumber(game.Pickup.GetDisplayDescriptionId(perkId))
      perkDescription = UI.GetLAMSMsg(perkDescription)
    end
    local attributeCount = 0
    local firstAttr, secondAttr
    local shouldDisplayBonuses = not runeUtil.HasFlag(runeID, "UniqueRune")
    local lamsHelper = function(stringName)
      local amount = runeUtil.GetRuneInfoImprovementAttributeValue(runeInfo, stringName)
      if amount ~= 0 then
        attributeCount = attributeCount + 1
        if secondAttr ~= nil then
          shouldDisplayBonuses = false
        elseif firstAttr == nil then
          firstAttr = lamsIDs[stringName .. "_Icon"]
        elseif secondAttr == nil then
          secondAttr = lamsIDs[stringName .. "_Icon"]
        end
      end
    end
    lamsHelper("Strength")
    lamsHelper("Runic")
    lamsHelper("Defense")
    lamsHelper("Resistance")
    lamsHelper("Focus")
    lamsHelper("Luck")
    if 5 < attributeCount then
      perkDescription = util.GetLAMSMsg(lamsIDs["6Bonus"])
    end
    local finalLamsMsg = perkDescription or ""
    if shouldDisplayBonuses then
      if secondAttr then
        finalLamsMsg = finalLamsMsg .. " " .. util.FormatLAMSString(lamsIDs["2Bonus"], util.GetLAMSMsg(firstAttr), util.GetLAMSMsg(secondAttr))
      elseif firstAttr then
        finalLamsMsg = finalLamsMsg .. " " .. util.FormatLAMSString(lamsIDs["1Bonus"], util.GetLAMSMsg(firstAttr))
      end
    end
    return {
      DisplayName = runeUtil.GetRuneDisplayNameFromID(runeID),
      SubtitleText = util.GetLAMSMsg(lamsIDs[rarity], rarity) .. " " .. typeName,
      Description = finalLamsMsg,
      IconText = runeUtil.GetRuneIconFromID(runeID),
      MaterialSwap = nil,
      RarityAnimationTime = RarityIndexToAnimTime(rarityIndex)
    }
  end
  local GetPickupDisplayInfo = function(resourceName)
    local rarity, rarityIndex = resourceUtil.GetRarity(resourceName)
    local slotName = pickupUtil.GetSlotName(resourceName)
    local typeName = pickupUtil.GetSlotDisplayName(slotName)
    if slotName == pickupConsts.Slot_WeaponComponent then
      typeName = util.GetLAMSMsg(lamsIDs.AxePommel)
    end
    local subTitleText = rarity == nil and typeName or util.GetLAMSMsg(lamsIDs[rarity], rarity) .. " " .. typeName
    local shouldUseMaterialSwap = pickupUtil.ShouldUseMaterialSwap(resourceName)
    local iconText, materialSwap
    if shouldUseMaterialSwap then
      materialSwap = resourceUtil.GetMaterialSwapName(resourceName)
    else
      iconText = resourceUtil.GetIconText(resourceName)
    end
    return {
      DisplayName = resourceUtil.GetDisplayName(resourceName),
      SubtitleText = subTitleText,
      Description = resourceUtil.GetDescription(resourceName),
      IconText = iconText,
      MaterialSwap = materialSwap,
      RarityAnimationTime = RarityIndexToAnimTime(rarityIndex)
    }
  end
  local GetResourceDisplayInfo = function(resourceName)
    local rarity, rarityIndex = resourceUtil.GetRarity(resourceName)
    local typeName = resourceName
    return {
      DisplayName = resourceUtil.GetDisplayName(resourceName),
      SubtitleText = util.GetLAMSMsg(lamsIDs.NoColon[rarity], rarity),
      Description = resourceUtil.GetDescription(resourceName),
      IconText = resourceUtil.GetIconText(resourceName),
      MaterialSwap = nil,
      RarityAnimationTime = RarityIndexToAnimTime(rarityIndex)
    }
  end
  local GetUpgradeDisplayInfo = function(resourceName)
    local rarity, rarityIndex = resourceUtil.GetRarity(resourceName)
    local upgradeSuccessfulBody = ""
    local shouldUseMaterialSwap = pickupUtil.ShouldUseMaterialSwap(resourceName)
    local iconText, materialSwap
    if shouldUseMaterialSwap then
      materialSwap = resourceUtil.GetMaterialSwapName(resourceName)
    else
      iconText = resourceUtil.GetIconText(resourceName)
    end
    if pickupUtil.IsWeapon(resourceName) then
      local isBlades = resourceName == pickupConsts.Blades
      local currStage = pickupUtil.GetProfileStage(resourceName)
      local maxStage = pickupUtil.GetMaxStage(resourceName)
      if not isBlades and currStage == maxStage then
        upgradeSuccessfulBody = util.FormatLAMSString(lamsIDs.UpgradeSuccessfulBody, resourceUtil.GetDisplayName(resourceName))
      else
        upgradeSuccessfulBody = util.FormatLAMSString(lamsIDs.UpgradeSuccessfulBodyWeapon, resourceUtil.GetDisplayName(resourceName))
      end
    else
      upgradeSuccessfulBody = util.FormatLAMSString(lamsIDs.UpgradeSuccessfulBody, resourceUtil.GetDisplayName(resourceName))
    end
    return {
      DisplayName = util.GetLAMSMsg(lamsIDs.UpgradeSuccessfulHeader),
      SubtitleText = "",
      Description = upgradeSuccessfulBody,
      IconText = iconText,
      MaterialSwap = materialSwap,
      RarityAnimationTime = RarityIndexToAnimTime(rarityIndex)
    }
  end
  local itemCardConsts = {
    DEFAULT = 0,
    SPECIAL = 1,
    TALISMAN = 2,
    POMMEL = 3,
    RUNE = 4,
    UPGRADE_SPECIAL = 5,
    UPGRADE_TALISMAN = 6,
    UPGRADE_POMMEL = 7
  }
  return {
    Consts = itemCardConsts,
    Icons = {
      [itemCardConsts.SPECIAL] = "itemCardMessage_ToggleRuneSpecialOn",
      [itemCardConsts.TALISMAN] = "itemCardMessage_ToggleTalismanOn",
      [itemCardConsts.POMMEL] = "itemCardMessage_ToggleSquareOn",
      [itemCardConsts.RUNE] = "itemCardMessage_ToggleEnchantmentOn",
      [itemCardConsts.DEFAULT] = "itemCardMessage_ToggleSquareOn",
      [itemCardConsts.UPGRADE_SPECIAL] = "itemCardMessage_ToggleRuneSpecialOn",
      [itemCardConsts.UPGRADE_TALISMAN] = "itemCardMessage_ToggleTalismanOn",
      [itemCardConsts.UPGRADE_POMMEL] = "itemCardMessage_ToggleSquareOn"
    },
    GetDisplayInfo = {
      [itemCardConsts.SPECIAL] = GetPickupDisplayInfo,
      [itemCardConsts.TALISMAN] = GetPickupDisplayInfo,
      [itemCardConsts.POMMEL] = GetPickupDisplayInfo,
      [itemCardConsts.RUNE] = GetRuneDisplayInfo,
      [itemCardConsts.DEFAULT] = GetResourceDisplayInfo,
      [itemCardConsts.UPGRADE_SPECIAL] = GetUpgradeDisplayInfo,
      [itemCardConsts.UPGRADE_TALISMAN] = GetUpgradeDisplayInfo,
      [itemCardConsts.UPGRADE_POMMEL] = GetUpgradeDisplayInfo
    }
  }
end
function DesignerMessage:SetItemCardMessage(messageRequest)
  assert(messageRequest ~= nil, "SetItemCardMessage called without a message request!")
  local itemCardArgs = messageRequest:get_itemCardArgs()
  assert(itemCardArgs ~= nil, "SetItemCardMessage called without item card data!")
  local resourceName = itemCardArgs.ResourceName
  local resourceType = itemCardArgs.ResourceType
  assert(not util.IsStringNilOrEmpty(resourceName), "Invalid resource given to SetItemCardMessage!")
  local goMessageRequestChild = messageRequest:get_goMessageRequestChild()
  local thName = messageRequest:GetTextHandle("Name")
  local thSubtitle = messageRequest:GetTextHandle("Subtitle")
  local thDesc = messageRequest:GetTextHandle("Desc")
  local helperType
  if resourceType == resourceConsts.RESOURCE_FLAG_RUNE then
    helperType = self.itemCardHelper.Consts.RUNE
  else
    local isPickup = resourceUtil.IsResourceAPickup(resourceName)
    if isPickup == true then
      local slotName = pickupUtil.GetSlotName(resourceName)
      if slotName == pickupConsts.Slot_ArmorTrinket then
        helperType = self.itemCardHelper.Consts.TALISMAN
      elseif slotName == pickupConsts.Slot_WeaponComponent or slotName == pickupConsts.Slot_WeaponComponent_Blades then
        helperType = self.itemCardHelper.Consts.POMMEL
      else
        helperType = self.itemCardHelper.Consts.SPECIAL
      end
      if resourceType == "Upgrade" then
        if helperType == self.itemCardHelper.Consts.SPECIAL then
          helperType = self.itemCardHelper.Consts.UPGRADE_SPECIAL
        elseif helperType == self.itemCardHelper.Consts.POMMEL then
          helperType = self.itemCardHelper.Consts.UPGRADE_POMMEL
        else
          helperType = self.itemCardHelper.Consts.UPGRADE_TALISMAN
        end
      end
    else
      helperType = self.itemCardHelper.Consts.DEFAULT
    end
  end
  local displayTable = self.itemCardHelper.GetDisplayInfo[helperType](resourceName)
  if messageRequest:get_formattedArgs() and string.match(displayTable.Description, "MSG:") ~= nil then
    local msg = util.GetLAMSStringFromMsg(displayTable.Description)
    displayTable.Description = string.format(msg, unpack(messageRequest:get_formattedArgs()))
  end
  UI.SetText(thName, displayTable.DisplayName)
  UI.SetText(thSubtitle, displayTable.SubtitleText)
  UI.SetText(thDesc, displayTable.Description)
  for _, icon in pairs(self.itemCardHelper.Icons) do
    UI.Anim(goMessageRequestChild, consts.AS_Forward, icon, 0, 0)
  end
  UI.Anim(goMessageRequestChild, consts.AS_Forward, self.itemCardHelper.Icons[helperType], 1, 1)
  UI.Anim(goMessageRequestChild, consts.AS_ForwardCycle, "itemCardMessage_Looping", 0.1, 0)
  UI.Anim(goMessageRequestChild, consts.AS_Forward, "itemCardMessage_Color", 0, displayTable.RarityAnimationTime)
  local goItemIcon = goMessageRequestChild:FindSingleGOByName("ItemIcon")
  local goUpgradeIcon = goMessageRequestChild:FindSingleGOByName("UpgradeIcon")
  if displayTable.IconText ~= nil then
    local thItemIcon = util.GetTextHandle(goItemIcon)
    UI.SetText(thItemIcon, displayTable.IconText)
    goItemIcon:Show()
    UI.Anim(goItemIcon, consts.AS_Forward, "default")
    goUpgradeIcon:Hide()
  elseif displayTable.MaterialSwap ~= nil then
    goUpgradeIcon:SetMaterialSwap(displayTable.MaterialSwap)
    goUpgradeIcon:Show()
    UI.Anim(goUpgradeIcon, consts.AS_Forward, "default")
    goItemIcon:Hide()
  end
end
function DesignerMessage:EVT_PAUSE_MENU_OPEN()
  self:WantPadEvents(false)
  self.pauseMenuOpen = true
end
function DesignerMessage:EVT_GLOBAL_MENU_CLOSED()
  if self.pauseMenuOpen == true then
    self:WantPadEvents(self.wantedPadEvents)
    self.pauseMenuOpen = false
    if self.currentMessage == nil then
      self:OnMessageComplete()
    end
  end
end
function DMTextOff:Setup()
  self.DM = self:GetState("designerMessage")
end
function DMTextOff:Enter()
  self.DM:HandleMessageExit()
end
function DMTextOn:Setup()
  self.DM = self:GetState("designerMessage")
end
function DMTextOn:UpdatePosition()
  local message = self.DM.currentMessage:get_goMessageRequest()
  if message == self.DM.goSidebarMessageSmall or message == self.DM.goSidebarMessageMed or message == self.DM.goSidebarMessageLarge then
    UI.SetGOScreenRight(message)
  end
end
function DMTextOn:Enter()
  self.DM.currentMessage = self.DM.messageQueue:Dequeue()
  self:UpdatePosition()
  self.DM:HandleMessageDelay()
end
function DMTextOn:EVT_UPDATE_WINDOW_SIZE()
  self:UpdatePosition()
end
function DMAdvancePressButton:Setup()
  self.DM = self:GetState("designerMessage")
end
function DMAdvancePressButton:Enter()
  self.DM:WantPadEvents(not self.pauseMenuOpen)
  self.DM.wantedPadEvents = true
  local nextMessage = self.DM.messageQueue:Peek()
  local nextMessageAdvanceTypeInfo = nextMessage and self.DM.advanceTypeInfo[nextMessage:get_advanceType()]
  self.DM.currentMessage:Activate(nextMessage ~= nil and nextMessageAdvanceTypeInfo ~= nil and (nextMessageAdvanceTypeInfo.PausesGame or nextMessageAdvanceTypeInfo.MasksButtons))
end
function DMAdvancePressButton:Exit()
  self.DM.currentMessage:Deactivate()
  self.DM:WantPadEvents(false)
  self.DM.wantedPadEvents = false
end
function DMAdvanceHoldButton:Setup()
  self.DM = self:GetState("designerMessage")
end
function DMAdvanceHoldButton:Enter()
  self:WantPadEvents(not self.pauseMenuOpen)
  self.wantedPadEvents = true
  local promptAdvancer = self.DM.currentMessage:get_sliderHoldFill()
  if promptAdvancer ~= nil then
    promptAdvancer:SetOnComplete(function()
      self:AdvanceDialog()
    end)
    promptAdvancer:Activate(true)
  end
end
function DMAdvanceHoldButton:Exit()
  local promptAdvancer = self.DM.currentMessage:get_sliderHoldFill()
  if promptAdvancer ~= nil then
    promptAdvancer:Deactivate(true)
  end
  self.DM.currentMessage:Deactivate()
  self:WantPadEvents(false)
  self.wantedPadEvents = false
end
function DMAdvanceHoldButton:AdvanceDialog()
  self:Goto("DMTextOff")
end
function DesignerMessage:OnSaveCheckpoint(tab)
end
function DesignerMessage:OnRestoreCheckpoint(tab)
end
local __MessageTable = {
  MaxRageUpgradePiece = {
    {
      MessageVariant = uiCalls.msgParam.SPLASHSCREEN_LANDSCAPE_SMALL,
      Title = 8264,
      Text = 40875,
      ImageMaterialName = "bloodmead",
      AdvanceType = uiCalls.msgParam.ADVANCE_PRESS
    }
  },
  MaxHealthUpgradePiece = {
    {
      MessageVariant = uiCalls.msgParam.SPLASHSCREEN_LANDSCAPE_SMALL,
      Title = 8262,
      Text = 40874,
      ImageMaterialName = "apple",
      AdvanceType = uiCalls.msgParam.ADVANCE_PRESS
    }
  },
  MaxRageUpgrade = {
    {
      MessageVariant = uiCalls.msgParam.SPLASHSCREEN_LANDSCAPE_SMALL,
      Title = 40093,
      Text = 40094,
      ImageMaterialName = "bloodmead",
      AdvanceType = uiCalls.msgParam.ADVANCE_PRESS
    }
  },
  MaxHealthUpgrade = {
    {
      MessageVariant = uiCalls.msgParam.SPLASHSCREEN_LANDSCAPE_SMALL,
      Title = 26048,
      Text = 26047,
      ImageMaterialName = "apple",
      AdvanceType = uiCalls.msgParam.ADVANCE_PRESS
    }
  },
  MuspelheimCipherComplete = {
    {
      MessageVariant = uiCalls.msgParam.SPLASHSCREEN_LANDSCAPE_SMALL,
      Title = 43238,
      Text = 43239,
      ImageMaterialName = "alf5",
      AdvanceType = uiCalls.msgParam.ADVANCE_PRESS,
      formattedArgs = {
        util.GetLAMSMsg(lamsIDs.Muspelheim),
        util.GetLAMSMsg(lamsIDs.Muspelheim)
      }
    }
  },
  NiflheimCipherComplete = {
    {
      MessageVariant = uiCalls.msgParam.SPLASHSCREEN_LANDSCAPE_SMALL,
      Title = 43238,
      Text = 43239,
      ImageMaterialName = "hel5",
      AdvanceType = uiCalls.msgParam.ADVANCE_PRESS,
      formattedArgs = {
        util.GetLAMSMsg(lamsIDs.Niflheim),
        util.GetLAMSMsg(lamsIDs.Niflheim)
      }
    }
  },
  SkapSlag = {
    {
      MessageVariant = uiCalls.msgParam.MESSAGE_BOTTOM,
      Text = 44804,
      DisplayTime = 5,
      Critical = true
    }
  }
}
function DesignerMessage:EVT_HandleResourceInitialized(name)
  local msg = __MessageTable[name]
  if msg ~= nil then
    self:StartTimer("__ResourceInitHelper__" .. name, 0.5, function()
      for _, realMsg in ipairs(msg) do
        if realMsg.MessageVariant == uiCalls.msgParam.MESSAGE_BOTTOM then
          uiCalls.SendBottomDesignerMessage(realMsg)
        else
          uiCalls.SendSplashScreenDesignerMessage(realMsg)
        end
      end
    end)
  end
end
function DesignerMessage:EVT_NewGamePlusMessage()
  local footerInfo
  if game.UI.GetInScreen(consts.SHELL) then
    footerInfo = {
      {
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Close),
        EventHandlers = {
          {
            Events = {
              "EVT_Advance_Release"
            },
            Handler = function()
              UI.SendEvent("*", consts.EVT_SHELL_NEW_GAME_PLUS_PROMPT_FINISHED)
            end
          }
        }
      }
    }
  else
    footerInfo = {
      {
        Text = "[StartButton] " .. util.GetLAMSMsg(lamsIDs.Options),
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release"
            },
            Handler = nil,
            UICalls = {
              {
                CallName = "UI_Event_Go_To_Settings",
                ArgTable = {
                  false,
                  false,
                  "NG+"
                }
              }
            }
          }
        }
      }
    }
  end
  uiCalls.SendSplashScreenDesignerMessage({
    Title = lamsIDs["NewGame+"],
    Text = 44735,
    AdvanceType = uiCalls.msgParam.ADVANCE_PRESS,
    MessageVariant = uiCalls.msgParam.SPLASHSCREEN_NO_IMAGE,
    FooterButtonInfo = footerInfo,
    Critical = true
  })
end
